function [CopParams, Rt, veclRt, LogL, OptInfo]=EstimateTVTCopula(data,corrspec,optimizer,method)

%%%%%%%%%%%%%% Estimation of time - varying T Copula %%%%%%%%%%%%%%%

% This function estimates the parameters of the T Copula. Three
% different specifications are assumed for the time - varying correlation
% equation, namely, DCC(1,1), TVC(1,1) and Patton's specification.
% Inputs:
% data:         A matrix of uniform U(0,1) or iid margins
% corrspec:     A string with values 'DCC', 'TVC' or 'Patton'
% optimizer:    A string with values 'fmincon' or 'fminunc'. If you choose
%               'Patton' in corrspec only fminunc is applied.
% method:       A string with values 'CML' or 'IMF'. If your data is U(0,1)
%               method has no meaning. If you input iid margins the 'CML'
%               transforms them to uniform with the empiricalCDF function,
%               while if you choose 'IMF' the transformation is being made
%               with tcdf
% Defaults:     default values for inputs are: 'DCC', 'fmincon' and 'CML'
% Warnings:     Since tinv(0) or tinv(1) are not real numbers you are
%               advised to always choose 'CML' for method
% Outputs:      
% CopParams:    The estimated parameters
% Rt:           3d object with time varying correlations
% veclRt:       Each column of veclRt contains the correlations of a single
%               pair from data
% LogL:         The neggative log likelihood value at the optimum
% OptInfo:      Structure that contains info about the optimization such as
%               the exitflag and the numerical standard errors. (Only if
%               you choose 'fminunc' as optimizer)
% ------------------------------------------------------------------------
% Author: Manthos Vogiatzoglou, UoM, 2008 - 2009
% contact at: vogia@yahoo.com

if nargin==3
    method='CML';
end
if nargin==2
    method='CML'; optimizer='fmincon';
end
if nargin==1
    method='CML'; optimizer='fmincon'; corrspec='DCC';
end
if nargin==4
    disp='notify';
else
    disp='iter';
end
[T,N]=size(data);
lower =[2.01;.001*ones(2,1)]; 
upper =[200; ones(2,1)];
A=[0 ones(1,2)];
b=0.9999;
options = optimset('Algorithm','active-set','Display',disp,'MaxFunEvals',10000,'TolCon',10^-12,'TolFun',10^-4,'TolX',10^-6);
if strcmp(corrspec,'DCC')==1 || strcmp(corrspec,'TVC')==1
    if strcmp(optimizer,'fmincon')==1
    initialparams=[8;.03;.95];
    [CopParams, likhood,exitflag]= fmincon('TCopulaLogLikhood',initialparams,A,b,[],[],lower,upper,[],options,data,corrspec,optimizer,method);
    elseif strcmp(optimizer,'fminunc')==1
    initialparams=[2;2;10];
    [CopParams, likhood,exitflag,output,grad,hessian]= fminunc('TCopulaLogLikhood',initialparams,options,data,corrspec,optimizer,method);
    if exitflag<0
    display('fminunc cannot find an optimum, fmincon is loaded')
    initialparams=[8;.03;.95];
    [CopParams, likhood,exitflag]= fmincon('TCopulaLogLikhood',initialparams,A,b,[],[],lower,upper,[],options,data,corrspec,optimizer,method);
    end
    OptInfo.exitflag=exitflag;
    OptInfo.grad=grad;
    OptInfo.Hessian=hessian;
    ARCHparam=CopParams(2)/(1+CopParams(2)+CopParams(3));
    GARCHparam=CopParams(3)/(1+CopParams(2)+CopParams(3));
    OptInfo.ConstrParams=[ARCHparam;GARCHparam];
    stder=LogLstdErrors(grad,hessian);
    OptInfo.NumStdErrors=stder;
    end
elseif strcmp(corrspec,'Patton')==1 && N==2
    samplecorr=corr(data);
    rhobar=samplecorr(1,2);
    initialparams = [8;log((1+rhobar)/(1-rhobar));0;0];
    if strcmp(optimizer,'fminunc')==1
    [CopParams, likhood,exitflag,output,grad,hessian]= fminunc('TCopulaLogLikhood',initialparams,options,data,corrspec,'fminunc',method);
        if exitflag>0
        OptInfo.exitflag=exitflag;
        OptInfo.grad=grad;
        OptInfo.Hessian=hessian;
        stder=LogLstdErrors(grad,hessian);
        OptInfo.NumStdErrors=stder;
        else
        display('fminunc cannot find an optimum, fmincon is loaded')
        lower =[2.01;-5*ones(3,1)]; 
        upper =[200; 5*ones(3,1)];
        [CopParams, likhood,exitflag]= fmincon('TCopulaLogLikhood',initialparams,[],[],[],[],lower,upper,[],options,data,corrspec,'fmincon',method);
        OptInfo.exitflag=exitflag;
        end
    elseif strcmp(optimizer,'fmincon')==1
    lower =[2.01;-5*ones(3,1)]; 
    upper =[200; 5*ones(3,1)];
    [CopParams, likhood,exitflag]= fmincon('TCopulaLogLikhood',initialparams,[],[],[],[],lower,upper,[],options,data,corrspec,'fmincon',method);
    OptInfo.exitflag=exitflag;
    end
end
[LogL,Rt,veclRt]=TCopulaLogLikhood(CopParams,data,corrspec,optimizer,method);
LogL=-LogL;
OptInfo.exitflag=exitflag;

